/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Id: Main.c 3194 2011-12-14 16:36:28Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2011-12-14 17:36:28 +0100 (Mi, 14 Dez 2011) $
    $Revision: 3194 $

   Targets:
     linux    : yes

   Description:
    netXTransport Toolkit usage example

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      1        23.02.13    SD       initial version

**************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <termios.h>
#include <unistd.h>
#include <sys/stat.h>
#include "netXTransport.h"
#include "netxtransport_linux.h"
#include "cifXErrors.h"
#include "rcX_Public.h"


#define HOST_TO_LE32(a)   (a)
#define LE32_TO_HOST(a)   (a)

/* global definitions */
#define CIFX_DEV_NO        0              /* default device number                                 */
#define FIRMWARE_NAME      "CIFXECS.NXF"  /* default firmware name     (see TryFirmwareDownload()) */
#define FIRMWARE_PATH      "./"           /* default path to firmware  (see TryFirmwareDownload()) */

/* cifX demo application specific  */
void RuncifXDemo(void);
static int kbhit();

int main(void)
{
  struct CIFX_LINUX_INIT init = {0};
  int32_t lRet = NXT_NO_ERROR;
#ifdef TRANSPORT_DEBUG
  init.trace_level = 0xff;
  printf("Setting trace level of libnetxtransport to 0x%X\n", (uint32_t)init.trace_level);
  printf("NOTE: Trace level of 0xFF is recommended to use only for debugging purposes!\n");
#else
  init.trace_level = 0x0;
#endif

  /* initialize netxtransport library */
  cifXDriverInit( &init);

  RuncifXDemo();

  cifXDriverDeinit();

  return 0;
}

/*****************************************************************************/
/*! Retreives user console input
*   \return character                                                        */
/*****************************************************************************/
static int kbhit()
{
  struct termios oldt, newt;
  int    ch;
  int    oldf;
  int    iRet = 0;

  tcgetattr(STDIN_FILENO, &oldt);
  newt = oldt;
  newt.c_lflag &= ~(ICANON | ECHO);
  tcsetattr(STDIN_FILENO, TCSANOW, &newt);
  oldf = fcntl(STDIN_FILENO, F_GETFL, 0);
  fcntl(STDIN_FILENO, F_SETFL, oldf | O_NONBLOCK);

  ch = getchar();

  tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
  fcntl(STDIN_FILENO, F_SETFL, oldf);

  if(ch != EOF)
  {
    ungetc(ch, stdin);
    iRet = 1;
  }
  return iRet;
}

/*****************************************************************************/
/*! Copies firmware file into ppbBuffer
*   \param szFileName       Path to firmware file
*   \param ppbBuffer        Pointer that receives pointer to buffer
*   \param pulFileSize      Size of ppbBuffer
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t GetFirmware( char* szFileName, uint8_t** ppbBuffer, uint32_t* pulFileSize)
{
  int32_t  lRet       = -1;
  FILE*    hFile      = NULL;
  uint32_t ulFileSize = 0;
  size_t   ReadSize   = 0;

  if (NULL != (hFile = fopen( szFileName, "r")))
  {
    struct stat buf;

    stat( szFileName, &buf);
    ulFileSize = buf.st_size;

    *ppbBuffer = (uint8_t*)malloc(ulFileSize);
    if (0!=(ReadSize = fread(*ppbBuffer, 1, ulFileSize, hFile)))
    {
      *pulFileSize = ulFileSize;
      return 0;
    } else
    {
      printf("Error reading all data!\n");
    }
  } else
  {
    printf("Error opening firmware file %s\n", szFileName);
  }
  return lRet;
}

/*****************************************************************************/
/*! Checks if file exists on device
*   \param hChannel    Handle to an opened channel
*   \param szFileName  Name of the file
*   \return 1 if it exists                                                   */
/*****************************************************************************/
int CheckIfFileExists(CIFXHANDLE hChannel, char* szFileName)
{
  int32_t             lRet           = CIFX_NO_ERROR;
  int                 fFound         = 0;
  CIFX_DIRECTORYENTRY tDirectoryInfo = {0};

  if (CIFX_NO_ERROR == (lRet = xSysdeviceFindFirstFile(  hChannel, 0, &tDirectoryInfo, NULL, NULL)))
  {
    while(lRet == CIFX_NO_ERROR)
    {
      lRet = xSysdeviceFindNextFile( hChannel, 0, &tDirectoryInfo, NULL, NULL);

      if (strcmp(tDirectoryInfo.szFilename, szFileName) == 0)
      {
        fFound = 1;
        break;
      }
    }
  }
  return fFound;
}

/*****************************************************************************/
/*! Shows the error description
*   \param lError  Error value                                               */
/*****************************************************************************/
void ShowError( int32_t lError)
{
  if( lError != CIFX_NO_ERROR)
  {
    /* Read driver error description */
    char szError[1024] ={0};
    xDriverGetErrorDescription( lError,  szError, sizeof(szError));
    printf("Error: 0x%X, <%s>\r\n", lError, szError);
  }
}

/*****************************************************************************/
/*! Writes channel information to console
*   \param hDriver     Handle to the opened driver
*   \param ulBoardIdx  Number of the board                                   */
/*****************************************************************************/
void ShowChannelInfo( CIFXHANDLE hDriver, uint32_t ulBoardIdx)
{
  unsigned long ulChannelIdx  = 0;
  int32_t       lChannelRet   = CIFX_NO_ERROR;

  while(lChannelRet == CIFX_NO_ERROR)
  {
    /* Read all channel information from the given board */
    CHANNEL_INFORMATION tChannelInfo = {{0}};
    lChannelRet = xDriverEnumChannels(hDriver, ulBoardIdx, ulChannelIdx, sizeof(tChannelInfo), &tChannelInfo);
    if(lChannelRet != CIFX_NO_ERROR)
    {
      if ((lChannelRet != CIFX_NO_MORE_ENTRIES) && (lChannelRet != CIFX_INVALID_CHANNEL))
      {
        /* Show information */
        printf("Error during xDriverEnumChannels(): Channel %u\r\n", (unsigned int)ulChannelIdx);
        ShowError(lChannelRet);
        break;
      }
    } else
    {
      /* Show information */
      printf("  Channel%u Information:\r\n", (unsigned int)ulChannelIdx);
      printf("   Channel Error            : 0x%08X\r\n",  tChannelInfo.ulChannelError);
      printf("   Board Name               : %s\r\n",  tChannelInfo.abBoardName);
      printf("   Alias Name               : %s\r\n",  tChannelInfo.abBoardAlias);
      printf("   Device Nr.               : %lu\r\n",  (long unsigned int)tChannelInfo.ulDeviceNumber);
      printf("   Serial Nr.               : %lu\r\n",  (long unsigned int)tChannelInfo.ulSerialNumber);
      printf("   MBX Size                 : %lu\r\n",  (long unsigned int)tChannelInfo.ulMailboxSize);
      printf("   Firmware Name            : %s\r\n",  tChannelInfo.abFWName);
      printf("   Firmware Version         : %d.%d.%d Build %d\r\n",  tChannelInfo.usFWMajor, tChannelInfo.usFWMinor,tChannelInfo.usFWRevision, tChannelInfo.usFWBuild);
      printf("   Open Counter             : %lu\r\n",  (long unsigned int)tChannelInfo.ulOpenCnt);
      printf("   Put Packet Counter       : %lu\r\n",  (long unsigned int)tChannelInfo.ulPutPacketCnt);
      printf("   Get Packet Counter       : %lu\r\n",  (long unsigned int)tChannelInfo.ulGetPacketCnt);
      printf("   Number of IO Input Areas : %lu\r\n",  (long unsigned int)tChannelInfo.ulIOInAreaCnt);
      printf("   Number of IO Output Areas: %lu\r\n",  (long unsigned int)tChannelInfo.ulIOOutAreaCnt);
      printf("   Size of handshake cells  : %lu\r\n",  (long unsigned int)tChannelInfo.ulHskSize);
      printf("   Actual netX Flags        : 0x%08X\r\n",tChannelInfo.ulNetxFlags);
      printf("   Actual host Flags        : 0x%08X\r\n",tChannelInfo.ulHostFlags);
    }
    ++ulChannelIdx; /* Next channel */
  }
}

/*****************************************************************************/
/*! Displays a hex dump on the debug console (16 bytes per line)
*   \param pbData     Pointer to dump data
*   \param ulDataLen  Length of data dump                                    */
/*****************************************************************************/
void DumpData(unsigned char* pbData, unsigned long ulDataLen)
{
  unsigned long ulIdx;

  for(ulIdx = 0; ulIdx < ulDataLen; ++ulIdx)
  {
    if(0 == (ulIdx % 16))
      printf("\r\n");

    printf("%02X ", pbData[ulIdx]);
  }
  printf("\r\n");
}

/*****************************************************************************/
/*! Dumps a rcX packet to debug console
*   \param ptPacket Pointer to packed being dumped                           */
/*****************************************************************************/
void DumpPacket(CIFX_PACKET* ptPacket)
{
  printf("Dest   : 0x%08lX      ID   : 0x%08lX\r\n",(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulDest),  (long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulId));
  printf("Src    : 0x%08lX      Sta  : 0x%08lX\r\n",(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulSrc),   (long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulState));
  printf("DestID : 0x%08lX      Cmd  : 0x%08lX\r\n",(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulDestId),(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulCmd));
  printf("SrcID  : 0x%08lX      Ext  : 0x%08lX\r\n",(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulSrcId), (long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulExt));
  printf("Len    : 0x%08lX      Rout : 0x%08lX\r\n",(long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulLen),   (long unsigned int)HOST_TO_LE32(ptPacket->tHeader.ulRout));

  printf("Data:");
  DumpData(ptPacket->abData, HOST_TO_LE32(ptPacket->tHeader.ulLen));
}


/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t ChannelDemo(CIFXHANDLE hDrv, char* cifx_dev)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("---------- Communication Channel demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hChannel = NULL;
    lRet = xChannelOpen(hDrv, cifx_dev, 0, &hChannel);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening Channel!");

    } else
    {
      CHANNEL_INFORMATION tChannelInfo = {{0}};

      /* Channel successfully opened, so query basic information */
      if( CIFX_NO_ERROR != (lRet = xChannelInfo(hChannel, sizeof(CHANNEL_INFORMATION), &tChannelInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("Communication Channel Info:\r\n");
        printf("Device Number    : %lu\r\n",(long unsigned int)tChannelInfo.ulDeviceNumber);
        printf("Serial Number    : %lu\r\n",(long unsigned int)tChannelInfo.ulSerialNumber);
        printf("Firmware         : %s\r\n", tChannelInfo.abFWName);
        printf("FW Version       : %u.%u.%u build %u\r\n",
                tChannelInfo.usFWMajor,
                tChannelInfo.usFWMinor,
                tChannelInfo.usFWRevision,
                tChannelInfo.usFWBuild);
        printf("FW Date          : %02u/%02u/%04u\r\n",
                tChannelInfo.bFWMonth,
                tChannelInfo.bFWDay,
                tChannelInfo.usFWYear);

        printf("Mailbox Size     : %lu\r\n",(long unsigned int)tChannelInfo.ulMailboxSize);
      }
      uint32_t    ulSendPktCount = 0;
      uint32_t    ulRecvPktCount = 0;
      CIFX_PACKET tSendPkt       = {{0}};
      CIFX_PACKET tRecvPkt       = {{0}};

      printf("\nStart put/get packet Demo!\n");

      /* Read Security EEPROM zone 1*/
      xChannelGetMBXState( hChannel, (uint32_t*)&ulRecvPktCount, (uint32_t*)&ulSendPktCount);
      printf("Channel Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
             ulSendPktCount, ulRecvPktCount);

      /* Do a basic Packet Transfer */
      if(CIFX_NO_ERROR != (lRet = xChannelPutPacket(hChannel, &tSendPkt, 10)))
      {
        printf("Error sending packet to device (0x%X)!\r\n", lRet);
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        if(CIFX_NO_ERROR != (lRet = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device (lRet=0x%X)!\r\n", lRet);
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);
        }
      }
      xChannelClose(hChannel);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", (unsigned int)lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;

}


/*****************************************************************************/
/*! Packet handling demo
*   \param hDriver      Handle to the opened driver
*   \param szBoardName  Name of the board                                    */
/*****************************************************************************/
void PacketDemo(CIFXHANDLE hDriver, char* szBoardName)
{
  int32_t         lRet        = CIFX_NO_ERROR;
  CIFXHANDLE      hSysdevice  = NULL;
  CIFX_PACKET     tSendPacket = {{0}};
  CIFX_PACKET     tRecvPacket = {{0}};
  uint32_t        ulVisible   = 5;
  uint32_t        ulLoopCnt   = 0;
  struct timespec tTimeout    = {0, 1000000};/* 1ms */
  uint8_t         stop_demo   = 0;

  printf("\nRunning test on system device...\n");

  if(CIFX_NO_ERROR == (xSysdeviceOpen(hDriver, szBoardName, &hSysdevice)))
  {
    NETX_SYSTEM_STATUS_BLOCK tSystemSatusBlock;
    uint32_t                 failedRecv = 0;
    uint32_t                 failedSend = 0;

    if (CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSysdevice,
                           CIFX_INFO_CMD_SYSTEM_STATUS_BLOCK,
                           sizeof(tSystemSatusBlock),
                           &tSystemSatusBlock)))
    {
      printf("Failed xSysdeviceInfo() request (lRet = 0x%X)\n", lRet);
    }
    memset(&tSendPacket.tHeader, 0, sizeof(tSendPacket.tHeader));
    memset(&tRecvPacket.tHeader, 0, sizeof(tRecvPacket.tHeader));

    tSendPacket.tHeader.ulCmd = RCX_HW_IDENTIFY_REQ;

    printf("\n**************************************************************\n");
    printf("Starting demo - Packet-Communication...\r\n");
    printf("**************************************************************\n");
    printf("Doing sequently xSysdevicePutPacket() - xSysdeviceGetPacket().\n");
    printf("\nTo schedule a reset, press 'r'...\n");
    printf("To show statistics, press 's'...\n");
    printf("To interrupt demo press 'q'...\n\n");
    printf("Packet configured for hw-identify (RCX_HW_IDENTIFY_REQ)\n");
    printf("xSysdevicePutPacket(): ->\n");
    printf("xSysdeviceGetPacket(): <-\n\n");

    do
    {
      uint8_t delay_counter;

      /* send packet */
      if (CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSysdevice, &tSendPacket, CIFX_TO_SEND_PACKET)))
      {
        printf("xSysdevicePutPacket() failed for packet Paket-ID=%d (lRet=0x%X)\n", tSendPacket.tHeader.ulId, lRet);
        sleep(1);
        failedSend++;

      } else
      {
        /* do not print every packet */
        if (ulLoopCnt < ulVisible)
        {
          printf("-> ulCmd=0x%08X, ulId=%d, lRet=0x%08X\n",
              tSendPacket.tHeader.ulCmd, tSendPacket.tHeader.ulId, lRet);
        }

        nanosleep( &tTimeout, NULL);

        /* get packet */
        if (CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSysdevice, sizeof(tRecvPacket), &tRecvPacket, CIFX_TO_SEND_PACKET)))
        {
          printf("xSysdeviceGetPacket() failed for packet Paket-ID=%d (lRet=0x%X)\n", tSendPacket.tHeader.ulId, lRet);
          sleep(1);
          failedRecv++;
        } else
        {
          /* do not print every packet */
          if (ulLoopCnt < ulVisible)
          {
            printf("<- ulCmd=0x%08X, ulId=%d, ulSta=0x%08X, ulLen=%d, lRet=0x%08X\n",
               tRecvPacket.tHeader.ulCmd, tRecvPacket.tHeader.ulId, tRecvPacket.tHeader.ulState, tRecvPacket.tHeader.ulLen, lRet);
          }
        }
      }
      if (ulLoopCnt == ulVisible)
        printf("\nSkip output (except errors and reset instruction)...\n");
      ++tSendPacket.tHeader.ulId;

      delay_counter=20;
      do {
        if (kbhit()) {
          uint8_t opt = getchar();
          if ('r' == opt) {
            printf("Resetting device...\n");
            if ((lRet = xSysdeviceReset( hSysdevice, 30000))) {
              printf("Error during reset!!! (lRet = 0x%X)\n", lRet);
              printf("Demo canceled...\n");
              break;
            }
            ulLoopCnt = 0;

            printf("Continue packet demo...\n");
          } else if ('s' == opt) {
            printf("Number of cycles\t\t: %d\n", ulLoopCnt);
            printf("Number of failed receive packets\t: %d\n", failedRecv);
            printf("Number of failed send packets\t\t: %d\n", failedSend);
          } else if ('q' == opt) {
            printf("Demo canceled by user...\n");
            stop_demo = 1;
            break;
          }
        }
        /* sleep only short see not below */
        if (lRet != CIFX_NO_ERROR)
          usleep(1000*50);

      /* NOTE: in case of an error (for example lost connection), its better to sleep otherwise */
      /* we may have 100% cpu load. so we sleep here just to be a little bit more responsive.   */
      } while((lRet != CIFX_NO_ERROR) && (delay_counter-->0));
      ulLoopCnt++;
    } while(stop_demo == 0);
    printf("\n");

    xSysdeviceClose(hSysdevice);
  } else
  {
    printf("Error open system device (lRet=0x%X)\n", lRet);
  }
}

/*****************************************************************************/
/*! Writes channel information to console
*   \param hDriver      Handle to the opened driver
*   \param szBoardName  Name of the board                                    */
/*****************************************************************************/
void TryFirmwareDownload( CIFXHANDLE hDriver, char* szBoardName)
{
  int32_t             lRet         = CIFX_NO_ERROR;
  CIFXHANDLE          hChannel     = NULL;
  uint8_t*            pabFileData  = NULL;
  uint32_t            ulFileSize   = 0;

  printf("\nTry to open channel 0 of device \"%s\"\n", szBoardName);
  if (CIFX_NO_ERROR == (lRet = xSysdeviceOpen( hDriver, szBoardName, &hChannel)))
  {
    if (CheckIfFileExists( hChannel, FIRMWARE_NAME))
    {
      printf("\nSkip firmware download. Firmware already existing!\n");
    } else
    {
      char* strFirmware = NULL;

      if (NULL != (strFirmware = malloc(strlen(FIRMWARE_PATH) + strlen(FIRMWARE_NAME)+1)))
      {
        sprintf( strFirmware, "%s%s", FIRMWARE_PATH, FIRMWARE_NAME);
        if (CIFX_NO_ERROR == GetFirmware( strFirmware, &pabFileData, &ulFileSize))
        {
          printf("\nTry to download the firmware\n");
          /* try firmware download */
          if (CIFX_NO_ERROR == (lRet = xSysdeviceDownload( hChannel,
                                     0,
                                     DOWNLOAD_MODE_FIRMWARE,
                                     FIRMWARE_NAME,
                                     pabFileData,
                                     ulFileSize,
                                     NULL,
                                     NULL,
                                     NULL)))
          {
            if (CheckIfFileExists( hChannel, FIRMWARE_NAME))
              printf("Successfully downloaded the firmware\n");
            else
              printf("Can not verify download result!\n");
          } else
          {
            printf("Error while downloading the firmware (lRet=0x%X)\n", lRet);
          }
        } else
        {
          printf("Error while reading firmware! Skip firmware download.\n");
        }
      } else
      {
        printf("Skip firmware download. Error while retrieving the firmware\n");
      }
    }
    xSysdeviceClose(hChannel);

  } else
  {
    printf("Error opening channel 0 (lRet=0x%X)\n", lRet);
  }
  if (pabFileData)
    free(pabFileData);
}

/*****************************************************************************/
/*! Enumerates a board given by the number.
*   \param hDriver      Handle to the opened driver
*   \param BoardNo      Handle to the opened driver
*   \param szBoardName  return the name of the board
*   return  CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t EnumBoards( CIFXHANDLE hDriver, uint32_t BoardNo, char* szBoardName)
{
  int32_t           lRet       = CIFX_NO_ERROR;
  BOARD_INFORMATION tBoardInfo = {0};

  /* discover boards */
  if (CIFX_NO_ERROR == (lRet = xDriverEnumBoards( hDriver, BoardNo, sizeof(tBoardInfo), &tBoardInfo)))
  {
    printf("Board%u Information:\r\n", (unsigned int)BoardNo);
    printf(" Name : %s\r\n", tBoardInfo.abBoardName);
    printf(" Alias: %s\r\n", tBoardInfo.abBoardAlias);
    printf(" DevNr: %lu\r\n", (long unsigned int)tBoardInfo.tSystemInfo.ulDeviceNumber);
    printf(" SN   : %lu\r\n", (long unsigned int)tBoardInfo.tSystemInfo.ulSerialNumber);
    printf("\r\n");

    strcpy( szBoardName, tBoardInfo.abBoardName);
  } else
  {
    if (lRet != CIFX_NO_MORE_ENTRIES) {
      if (lRet == CIFX_INVALID_BOARD) {
        printf("Board %d. not found (lRet=0x%X)\n", BoardNo, lRet);
      } else {
        printf("Error while enumerating available devices (lRet=0x%X)\n", lRet);
      }
    }
  }
  return lRet;
}

/*****************************************************************************/
/*! Starts cifX demo application                                             */
/*****************************************************************************/
void RuncifXDemo(void)
{
  CIFXHANDLE        hDriver  = NULL;
  int32_t           lRet     = CIFX_NO_ERROR;
  int               BoardNo  = CIFX_DEV_NO;
  int               run_demo = 1;
  char              szBoardName[256];

  if (CIFX_NO_ERROR != (lRet = xDriverOpen(&hDriver))) {
    printf("Error starting driver lRet=0x%X\n", lRet);
    printf("Stopping application...\n\n");
  }
  while((CIFX_NO_ERROR == lRet) && (run_demo != 0)) {
    /* open the driver */
    if (lRet == CIFX_NO_ERROR)
    {
      char               user_input[256];
      char               job         = 0;
      DRIVER_INFORMATION tDriverInfo = {0};

      if (CIFX_NO_ERROR != (lRet = xDriverGetInformation( hDriver, sizeof(tDriverInfo), &tDriverInfo)))
      {
        printf("Error while requesting the driver information (lRet=0x%X)!\n", lRet);
        printf("Stopping application...\n\n");
      } else
      {
        printf("Driver Version: %s\n", tDriverInfo.abDriverVersion);
        printf("Device Count  : %d\n\n", tDriverInfo.ulBoardCnt);

        if (tDriverInfo.ulBoardCnt == 0) {
          printf("!!! No boards found please check hardware (power, connection,...) and           !!!\n");
          printf("!!! interface configuration or try rescan in case device is not recognized yet. !!!\n\n");
        }
        BoardNo = 0;
        printf("Select one of the following options:\n");
        printf("------------------------------------\n");
        if (tDriverInfo.ulBoardCnt > 0) {
          printf("'0'...'%d': as valid board number\n", tDriverInfo.ulBoardCnt-1);
        }
        printf("      'r': rescan interfaces (update interface states)\n");
        printf("           Note: A rescan re-initializes the interfaces. This process may change the device order,\n");
        printf("                 since de-actived devices will be deleted as well as new devices will be found!\n");
        printf("      'q': stop application\n");
        if (0 != scanf("%s", user_input)) {
          BoardNo = atoi(user_input);
          if ((strlen(user_input) == 1) && ((user_input[0] == 'q') || (user_input[0] == 'r'))) {
            job = user_input[0];
          }
        }
        fflush(stdin);

        if (job >= 'q')  {
          if ((job == 'q') || (job == 'r')) {
            if (job == 'q') {
              printf("\nApplication stopped by user!\n");
              run_demo = 0;
            }
            xDriverClose( hDriver);
            hDriver = NULL;
            /* NOTE: check if rescan is asked:                                                           */
            /* Rescan or update of the interfaces is just xDriverClose() and xDriverOpen(). Otherwise we */
            /* still work with old states -> unplugged devices are encountered but marked as disabled    */
            if (job == 'r') {
              printf("\nRescan triggered...\n\n");
              lRet = xDriverOpen( &hDriver);
              if (lRet != CIFX_NO_ERROR) {
                printf("Error restarting driver lRet=0x%X\n", lRet);
                printf("Stopping application...\n\n");
                run_demo = 0;
                hDriver = NULL;
              }
            }
          } else {
            printf("\nPlease enter a valid command!\n\n");
          }
          continue;
        } else {
          if (tDriverInfo.ulBoardCnt == 0) {
            printf("\nPlease select a valid option!\n\n");
            continue;
          } else {
            printf("\nRun demo on board number %d...\n\n",BoardNo);
          }
        }

        /* if the first request fails,  */
        lRet = EnumBoards( hDriver, BoardNo, szBoardName);
        if (lRet == CIFX_NO_ERROR)
        {
          ShowChannelInfo( hDriver, BoardNo);

          TryFirmwareDownload( hDriver, szBoardName);
          PacketDemo( hDriver, szBoardName);
          ChannelDemo(hDriver, szBoardName);
        } else {
          if (lRet == CIFX_INVALID_BOARD) {
            printf("xDriverEnumBoards() returned CIFX_INVALID_BOARD, which probably means the device is temporarily de-activated.\n");
            printf("Please check the connection if the device is unplugged. Fixing the connection problem may re-activate the device!\n");
            printf("Fixing the connection problem and try to access again.\n\n");

            printf("NOTE: In case the connection problem remains a 'rescan' will delete the device and it will be not possible to re-active\n");
            printf("      the device, as a consequence the device index may change!\n\n");
            lRet = CIFX_NO_ERROR;
          }
          if (lRet == CIFX_NO_MORE_ENTRIES) {
            printf("xDriverEnumBoards() returned CIFX_NO_MORE_ENTRIES, which means requested board with index %d does not exist.\n", BoardNo);
            printf("\nPlease choose another board number!\n\n");
            lRet = CIFX_NO_ERROR;
          }
        }
      }
    }
  }
  if (hDriver != NULL)
    xDriverClose( hDriver);

  hDriver = NULL;
}
